<?php

namespace App\Http\Controllers;

use App\Models\Staff;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class StaffController extends Controller
{
    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string',
            'role' => 'required|string',
            'phone' => 'required|string',
            'password' => 'required|string|min:4',
            'specialties' => 'array',
            'is_vip' => 'boolean',
            'avatar_url' => 'nullable|string'
        ]);

        // If ID is provided, it's an update (or upsert if using string IDs)
        // But usually store is for create. Frontend seems to generate ID for new staff? 
        // Or we should let backend generate.
        
        $staff = Staff::create([
            'id' => 'staff-' . uniqid(),
            'name' => $data['name'],
            'role' => $data['role'],
            'phone' => $data['phone'],
            'specialties' => $data['specialties'] ?? [],
            'is_vip' => $data['is_vip'] ?? false,
            'avatar_url' => $data['avatar_url'] ?? '/images/assets/sara.png'
        ]);

        $user = User::where('phone', $data['phone'])->first();
        if (!$user) {
            $user = User::create([
                'phone' => $data['phone'],
                'name' => $data['name'],
                'role' => 'STAFF',
                'password' => Hash::make($data['password']),
            ]);
        } else {
            $user->name = $data['name'];
            $user->role = 'STAFF';
            $user->password = Hash::make($data['password']);
            $user->save();
        }

        return response()->json($staff);
    }

    public function update(Request $request, $id)
    {
        $staff = Staff::findOrFail($id);
        $oldPhone = $staff->phone;
        
        $data = $request->validate([
            'name' => 'string',
            'role' => 'string',
            'phone' => 'string',
            'password' => 'nullable|string|min:4',
            'specialties' => 'array',
            'is_vip' => 'boolean',
        ]);

        $password = $data['password'] ?? null;
        unset($data['password']);

        $staff->update($data);

        $newPhone = $data['phone'] ?? $oldPhone;
        $user = User::where('phone', $oldPhone)->first();
        if (!$user && $newPhone !== $oldPhone) {
            $user = User::where('phone', $newPhone)->first();
        }

        if (!$user) {
            $user = User::create([
                'phone' => $newPhone,
                'name' => $staff->name,
                'role' => 'STAFF',
                'password' => Hash::make($password ?? $newPhone),
            ]);
        } else {
            $user->phone = $newPhone;
            $user->name = $staff->name;
            $user->role = 'STAFF';
            if ($password) {
                $user->password = Hash::make($password);
            }
            $user->save();
        }

        return response()->json($staff);
    }
    
    public function destroy($id) {
        $staff = Staff::findOrFail($id);
        $staff->delete();
        return response()->json(['message' => 'Staff deleted']);
    }
}
