<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Schedule;
use App\Models\Booking;
use App\Models\Staff;
use Carbon\Carbon;
use App\Support\DateHelper;

class ScheduleController extends Controller
{
    private function normalizeDigits($array) {
        if (!is_array($array)) return [];
        return array_map(function($item) {
            if (!is_string($item)) return $item;
            $persian = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
            $arabic  = ['٠', '١', '٢', '٣', '٤', '٥', '٦', '٧', '٨', '٩'];
            $english = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
            
            $item = str_replace($persian, $english, $item);
            $item = str_replace($arabic, $english, $item);
            return trim($item);
        }, $array);
    }

    /**
     * Get aggregated schedule for a specific date and target.
     * Logic: Global Template -> Global Override -> Staff Template (if any) -> Staff Override (if any).
     */
    public function getSchedule(Request $request)
    {
        $date = $request->input('date'); // Y-m-d
        if ($date) {
            $date = DateHelper::toGregorian($date);
        }
        $staffId = $request->input('staff_id'); // Optional
        
        $carbonDate = Carbon::parse($date);
        $dayOfWeek = $carbonDate->dayOfWeek; // 0 (Sunday) - 6 (Saturday) - NOTE: Laravel uses Carbon 0=Sun. Need to map to our system if needed. 
        // Our system frontend usually 0-6 or ISO. Let's assume standard 0(Sun)-6(Sat).

        // 1. Global Salon Template (staff_id=null, day_of_week=$dayOfWeek)
        $globalTemplate = Schedule::whereNull('staff_id')
            ->whereNull('date')
            ->where('day_of_week', $dayOfWeek)
            ->first();
        
        // 2. Global Salon Override (staff_id=null, date=$date)
        $globalOverride = Schedule::whereNull('staff_id')
            ->whereDate('date', $date)
            ->first();

        // Determine Base Salon Hours (Expanded to match frontend ALL_HOURS)
        $defaultHours = ['06:00', '07:00' ,'08:00', '09:00', '10:00', '11:00', '12:00', '13:00', '14:00', '15:00', '16:00', '17:00', '18:00', '19:00', '20:00', '21:00', '22:00', '23:00', '24:00'];
        
        $salonHours = $this->normalizeDigits($globalOverride ? $globalOverride->hours : ($globalTemplate ? $globalTemplate->hours : $defaultHours));
        $isSalonClosed = $globalOverride ? $globalOverride->is_holiday : ($globalTemplate ? $globalTemplate->is_holiday : false);

        \Log::info("Get Schedule Request: Date={$date}, StaffID={$staffId}, DayOfWeek={$dayOfWeek}, SalonClosed=".($isSalonClosed?'Yes':'No'));

        if ($isSalonClosed && ($staffId === 'GLOBAL' || !$staffId)) {
            return response()->json([
                'working_hours' => [],
                'hours' => [],
                'is_closed' => true,
                'is_salon_closed' => true,
                'salon_hours' => $salonHours,
                'reason' => 'Salon Closed (Global)'
            ]);
        }

        // If no staff specified, return Salon Hours
        if (!$staffId || $staffId === 'GLOBAL') {
            return response()->json([
                'working_hours' => $salonHours,
                'hours' => $salonHours,
                'salon_hours' => $salonHours,
                'is_closed' => false,
                'is_salon_closed' => false
            ]);
        }

        // 3. Staff Template (staff_id=$staffId, day_of_week=$dayOfWeek)
        $staffTemplate = Schedule::where('staff_id', $staffId)
            ->whereNull('date')
            ->where('day_of_week', $dayOfWeek)
            ->first();

        // 4. Staff Override (staff_id=$staffId, date=$date)
        $staffOverride = Schedule::where('staff_id', $staffId)
            ->whereDate('date', $date)
            ->first();

        $staffHours = $this->normalizeDigits($staffOverride ? $staffOverride->hours : ($staffTemplate ? $staffTemplate->hours : null));
        $isStaffClosed = $staffOverride ? $staffOverride->is_holiday : ($staffTemplate ? $staffTemplate->is_holiday : false);

        \Log::info("Staff Schedule: ID={$staffId}, Closed=".($isStaffClosed?'Yes':'No').", HoursCount=".(is_array($staffHours)?count($staffHours):'null'));

        if ($isStaffClosed) {
            return response()->json([
                'working_hours' => [],
                'hours' => [],
                'is_closed' => true,
                'is_staff_closed' => true,
                'reason' => 'Staff Off'
            ]);
        }

        // If Staff has defined hours, calculate intersection with Salon Hours
        if ($staffHours !== null && count($staffHours) > 0) {
            // Intersection Logic
            $finalHours = array_values(array_intersect($salonHours, $staffHours));
        } else {
            // If staff hasn't defined anything, assume they follow salon hours
            $finalHours = $salonHours;
        }

        // Fetch Detailed Bookings for Overlap Calculation
        $dayBookings = Booking::with('services')
            ->where('staff_id', $staffId)
            ->whereDate('date', $date)
            ->get()
            ->map(function($b) {
                return [
                    'id' => $b->id,
                    'time' => $b->time, // Bookings might be in Persian too? Better normalize them later in output if needed
                    'staffId' => $b->staff_id,
                    'serviceLineIds' => $b->services->pluck('id')->toArray(),
                    'totalDuration' => $b->services->sum('duration_minutes')
                ];
            });

        $finalResponse = [
            'working_hours' => $isSalonClosed ? [] : $finalHours,
            'booked_times' => $this->normalizeDigits($dayBookings->pluck('time')->toArray()),
            'day_bookings' => $dayBookings->map(function($db) {
                $db['time'] = str_replace(['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'], ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'], $db['time']);
                return $db;
            }),
            'hours' => array_values(array_diff($isSalonClosed ? [] : $finalHours, $this->normalizeDigits($dayBookings->pluck('time')->toArray()))), // Keep for backward compatibility
            'is_closed' => $isStaffClosed || $isSalonClosed,
            'is_staff_closed' => $isStaffClosed,
            'is_salon_closed' => $isSalonClosed,
            'raw_hours' => $staffHours, // Standardized to English
            'salon_hours' => $salonHours // Standardized to English
        ];

        \Log::info("Final Res Hours Count: ".count($finalHours).", isSalonClosed: ".($isSalonClosed?'Yes':'No'));
        return response()->json($finalResponse);
    }

    public function store(Request $request) {
        \Log::info('Schedule Store Request:', $request->all());
        try {
            if ($request->has('date')) {
                $request->merge(['date' => DateHelper::toGregorian($request->input('date'))]);
            }
            $data = $request->validate([
                'staff_id' => 'nullable|string',
                'date' => 'nullable|date_format:Y-m-d',
                'day_of_week' => 'nullable|integer|min:0|max:6',
                'hours' => 'nullable|array',
                'is_holiday' => 'boolean'
            ]);

            if (isset($data['hours'])) {
                $data['hours'] = $this->normalizeDigits($data['hours']);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            \Log::error('Validation Error:', $e->errors());
            throw $e;
        } catch (\Exception $e) {
            \Log::error('General Error:', ['msg' => $e->getMessage()]);
            return response()->json(['message' => $e->getMessage()], 500);
        }

        $query = Schedule::query();
        
        // Handle Staff ID (null or string)
        if (array_key_exists('staff_id', $data) && $data['staff_id'] !== 'GLOBAL' && $data['staff_id'] !== null) {
            $query->where('staff_id', $data['staff_id']);
        } else {
            $query->whereNull('staff_id');
        }

        // Handle Date (null or string)
        if (array_key_exists('date', $data) && $data['date'] !== null) {
            $query->whereDate('date', $data['date']);
        } else {
            $query->whereNull('date');
        }

        // Handle Day of Week (null or integer)
        if (array_key_exists('day_of_week', $data) && $data['day_of_week'] !== null) {
            $query->where('day_of_week', $data['day_of_week']);
        } else {
            $query->whereNull('day_of_week');
        }

        $schedule = $query->first();

        if ($schedule) {
            $schedule->update([
                'hours' => $data['hours'],
                'is_holiday' => $data['is_holiday'] ?? false
            ]);
        } else {
            $staffId = (array_key_exists('staff_id', $data) && $data['staff_id'] !== 'GLOBAL') ? $data['staff_id'] : null;
            Schedule::create([
                'staff_id' => $staffId,
                'date' => $data['date'] ?? null,
                'day_of_week' => $data['day_of_week'] ?? null,
                'hours' => $data['hours'],
                'is_holiday' => $data['is_holiday'] ?? false
            ]);
        }

        // CRITICAL FIX: If we are saving a WEEKDAY TEMPLATE (date is null, day_of_week is set),
        // we should clear any specific DATE OVERRIDES for that same weekday to ensure 
        // the new template takes effect immediately everywhere.
        if (is_null($data['date'] ?? null) && !is_null($data['day_of_week'] ?? null)) {
            $staffIdForClear = (array_key_exists('staff_id', $data) && $data['staff_id'] !== 'GLOBAL') ? $data['staff_id'] : null;
            
            $allOverrides = Schedule::where(function($q) use ($staffIdForClear) {
                if (is_null($staffIdForClear)) $q->whereNull('staff_id');
                else $q->where('staff_id', $staffIdForClear);
            })->whereNotNull('date')->get();

            foreach ($allOverrides as $override) {
                if (Carbon::parse($override->date)->dayOfWeek == $data['day_of_week']) {
                    $override->delete();
                }
            }
        }

        return response()->json(['message' => 'Schedule saved successfully']);
    }
}
